<?php
/**
 * Plugin Name: Taxa SGR PRO 2025
 * Plugin URI:  https://sgr-pro.ro
 * Description: Adaugă automat taxa SGR 0.50 lei pe produs + licență online + actualizări automate
 * Version:     1.5.3
 * Author:      SGR Romania
 * Author URI:  https://sgr-pro.ro
 * License:     GPL-2.0+
 * Text Domain: sgr-tax-manager
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 8.5
 * WC tested up to: 9.7
 */

if (!defined('ABSPATH')) exit;

// ================= CONFIGURARE (SCHIMBĂ DOAR ACESTE 3 LINII) =================
define('SGR_LICENSE_API', 'https://sgr.42web.io/verify.php');        // ← domeniul tău
define('SGR_UPDATE_URL',  'https://sgr.42web.io/update/info.json');  // ← domeniul tău
define('SGR_SECRET',      'alege-un-secret-lung-aici-2025xyz'); // ← același ca în config.php

// HPOS Compatibility
add_action('before_woocommerce_init', function () {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

// ================= VERIFICARE LICENȚĂ ONLINE =================
function sgr_check_license() {
    $key = trim(get_option('sgr_license_key', ''));
    if (empty($key)) return false;

    $cached = get_transient('sgr_license_status');
    if ($cached === 'valid') return true;
    if ($cached === 'invalid') return false;

    $domain = parse_url(home_url(), PHP_URL_HOST);
    $hash   = hash_hmac('sha256', $key . $domain, SGR_SECRET);

    $response = wp_remote_post(SGR_LICENSE_API, [
        'body'    => ['key' => $key, 'domain' => $domain, 'hash' => $hash],
        'timeout' => 15,
        'headers' => ['User-Agent' => 'SGR-Tax-Plugin/1.5.3']
    ]);

    if (is_wp_error($response)) return false;

    $body = json_decode(wp_remote_retrieve_body($response), true);
    $valid = !empty($body['valid']);

    set_transient('sgr_license_status', $valid ? 'valid' : 'invalid', WEEK_IN_SECONDS);
    return $valid;
}

// Blocare completă dacă licența nu e validă
if (!sgr_check_license() && !is_admin()) {
    return;
}

// ================= PLUGIN PRINCIPAL =================
class SGR_Tax_Manager {
    private static $instance = null;
    private $table;

    public static function init() {
        if (!self::$instance) self::$instance = new self();
        return self::$instance;
    }

    private function __construct() {
        global $wpdb;
        $this->table = $wpdb->prefix . 'sgr_tax_records';

        register_activation_hook(__FILE__, [$this, 'activate']);
        add_action('plugins_loaded', [$this, 'check_woocommerce']);
        add_action('init', [$this, 'load_textdomain']);

        if (is_admin()) {
            add_action('admin_menu', [$this, 'admin_menu']);
            add_action('admin_init', [$this, 'register_settings']);
            add_action('admin_notices', [$this, 'admin_notice']);
        }

        if (sgr_check_license()) {
            $this->enable_features();
        }
    }

    public function activate() {
        global $wpdb;
        $charset = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            order_id bigint(20) NOT NULL,
            sgr_amount decimal(10,2) NOT NULL,
            order_date datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id)
        ) $charset;";
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);

        add_option('sgr_tax_rate', '0.50');
        add_option('sgr_currency_symbol', 'lei');
    }

    public function check_woocommerce() {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', function () {
                echo '<div class="notice notice-error"><p>Taxa SGR PRO necesită WooCommerce activ!</p></div>';
            });
        }
    }

    public function load_textdomain() {
        load_plugin_textdomain('sgr-tax-manager', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    public function admin_notice() {
        if (sgr_check_license() || !current_user_can('manage_woocommerce')) return;
        echo '<div class="notice notice-error is-dismissible">
            <p><strong>Taxa SGR este dezactivată!</strong> 
            <a href="' . admin_url('admin.php?page=sgr-tax-manager') . '">Introdu codul de licență aici</a></p>
        </div>';
    }

    public function admin_menu() {
        add_menu_page(
            'Taxa SGR PRO',
            'Taxa SGR',
            'manage_woocommerce',
            'sgr-tax-manager',
            [$this, 'settings_page'],
            'dashicons-money-alt',
            58
        );
    }

    public function register_settings() {
        register_setting('sgr_settings', 'sgr_tax_rate', 'floatval');
        register_setting('sgr_settings', 'sgr_currency_symbol', 'sanitize_text_field');
    }

    public function settings_page() {
        if (isset($_POST['sgr_license_key'])) {
            update_option('sgr_license_key', sanitize_text_field($_POST['sgr_license_key']));
            delete_transient('sgr_license_status');
        }
        $valid = sgr_check_license();
        ?>
        <div class="wrap">
            <h1>Taxa SGR PRO 2025</h1>
            <?php echo $valid
                ? '<div class="notice notice-success"><p>Licență validă și activă!</p></div>'
                : '<div class="notice notice-error"><p>Licență invalidă sau lipsă!</p></div>'; ?>

            <form method="post">
                <table class="form-table">
                    <tr>
                        <th>Cod licență</th>
                        <td>
                            <input type="text" name="sgr_license_key" value="<?php echo esc_attr(get_option('sgr_license_key','')); ?>" 
                                   class="regular-text" placeholder="ex: SGR2025-PRO-ABC123" />
                            <p class="description">Se verifică automat online la fiecare 7 zile.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Verifică licența'); ?>
            </form>

            <hr><h2>Setări taxă SGR</h2>
            <form method="post" action="options.php">
                <?php settings_fields('sgr_settings'); ?>
                <table class="form-table">
                    <tr><th>Valoare taxă</th><td><input type="number" step="0.01" name="sgr_tax_rate" value="<?php echo esc_attr(get_option('sgr_tax_rate','0.50')); ?>" /> lei</td></tr>
                    <tr><th>Simbol</th><td><input type="text" name="sgr_currency_symbol" value="<?php echo esc_attr(get_option('sgr_currency_symbol','lei')); ?>" /></td></tr>
                </table>
                <?php submit_button('Salvează setările'); ?>
            </form>
        </div>
        <?php
    }

    public function enable_features() {
        // Tab în produs
        add_filter('woocommerce_product_data_tabs', function($tabs) {
            $tabs['sgr'] = ['label' => 'SGR', 'target' => 'sgr_data', 'class' => ['show_if_simple','show_if_variable']];
            return $tabs;
        });
        add_action('woocommerce_product_data_panels', function() {
            echo '<div id="sgr_data" class="panel woocommerce_options_panel"><div class="options_group">';
            woocommerce_wp_checkbox(['id' => '_sgr_enabled', 'label' => 'Activează taxa SGR pe acest produs']);
            echo '<p class="form-field"><small>Se adaugă automat 0.50 lei/bucată în coș</small></p></div></div>';
        });
        add_action('woocommerce_process_product_meta', function($id) {
            update_post_meta($id, '_sgr_enabled', isset($_POST['_sgr_enabled']) ? 'yes' : 'no');
        });

        // Adaugă taxa în coș
        add_action('woocommerce_cart_calculate_fees', function($cart) {
            if (is_admin() && !defined('DOING_AJAX')) return;
            $rate = floatval(get_option('sgr_tax_rate', 0.50));
            $count = 0;
            foreach ($cart->get_cart() as $item) {
                if (get_post_meta($item['product_id'], '_sgr_enabled', true) === 'yes') {
                    $count += $item['quantity'];
                }
            }
            if ($count > 0) {
                $cart->add_fee("Taxa SGR ({$count} produs" . ($count > 1 ? 'e' : '') . ")", $rate * $count, true);
            }
        });
    }
}
add_action('plugins_loaded', ['SGR_Tax_Manager', 'init']);

// ================= ACTUALIZĂRI AUTOMATE OTA =================
if (!class_exists('SGR_Updater')) {
    class SGR_Updater {
        private $file;
        public function __construct($file) {
            $this->file = $file;
            add_action('admin_init', [$this, 'init']);
        }
        public function init() {
            if (!function_exists('get_plugin_data')) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            $plugin = get_plugin_data($this->file);
            $basename = plugin_basename($this->file);

            add_filter('pre_set_site_transient_update_plugins', function($transient) use ($plugin, $basename) {
                if (empty($transient->checked)) return $transient;

                $remote = get_transient('sgr_update_check');
                if (false === $remote) {
                    $remote = wp_remote_get(SGR_UPDATE_URL, ['timeout' => 15]);
                    if (!is_wp_error($remote) && wp_remote_retrieve_response_code($remote) === 200) {
                        $remote = json_decode(wp_remote_retrieve_body($remote));
                        set_transient('sgr_update_check', $remote, 12 * HOUR_IN_SECONDS);
                    }
                }

                if ($remote && version_compare($plugin['Version'], $remote->version, '<')) {
                    $obj = new stdClass();
                    $obj->slug = $basename;
                    $obj->new_version = $remote->version;
                    $obj->url = $remote->details_url ?? '';
                    $obj->package = $remote->download_url;
                    $transient->response[$basename] = $obj;
                }
                return $transient;
            });
        }
    }
    new SGR_Updater(__FILE__);
}